/************************************************************************************************************\

Module Name:    LClient.h

Description:

References:

    Copyright (c) 2014, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LCLIENT_H
#define INC_LCLIENT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LStatus.h"
#include "Liberatus.h"

// -----------------------------------------------------------------------------------------------------------
//                                   U S A G E
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                   O V E R V I E W
// -----------------------------------------------------------------------------------------------------------
typedef MHANDLE     LClient_Handle;

#ifndef LCLIENTSERVERSHAREDENUMS
#define LCLIENTSERVERSHAREDENUMS


#define     LCLIENTSERVER_MAX_CLIENTS               8
#define     LCLIENTSERVER_MAX_SERVER                8
#define     LCLIENTSERVER_NO_TIMEOUT                0xFFFFFFFF

/************************************************************************************************************\

Enum:           LClientServer_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LClientServer_AttributeType_CREATE_CLIENT,          // Attributes structure for the client Create method.
    LClientServer_AttributeType_CREATE_SERVER,          // Attributes structure for the server Create method.
    LClientServer_AttributeType_FORCE32 = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.
} LClientServer_AttributeType;

/************************************************************************************************************\

Enum:           LClientServer_SubmissionType

Description:    Enumeration of submission types.

Comments:       - The various notification types share a common Submission structure. The interpretation of
                  the content is type specific.

\************************************************************************************************************/
typedef enum
{
    LClientServer_SubmissionType_HEADER,              // Header passed to the WaitSubmission method.
    LClientServer_SubmissionType_USER         = 256,  // User specific submission. Can be used with
                                                      // SendSubmission and received from Wait
    LClientServer_SubmissionType_FORCE32      = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.

} LClientServer_SubmissionType;


/************************************************************************************************************\

Enum:           LClientServer_NotificationType

Description:    Enumeration of notification types.

Comments:       - The various notification types share a common Notification structure. The interpretation of
                  the content is type specific.

\************************************************************************************************************/
typedef enum
{
    LClientServer_NotificationType_HEADER,              // Header passed to the WaitNotification method.
    LClientServer_NotificationType_ERROR,               // Generic error notification. Can be received from
    LClientServer_NotificationType_USER         = 256,  // User specific notification. Can be used with
                                                        // SendNotification and received from WaitNotification.
    LClientServer_NotificationType_FORCE32      = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.

} LClientServer_NotificationType;

/************************************************************************************************************\

Enum:           LClientServer_NotificationFlags

Description:    Enumeration of notification types.

Comments:       - These flags are returned in the oHeader.uFlags member of the Notification structure.
                  The flags correspond to independent bit if a bitmask and can be set simultaneously.

\************************************************************************************************************/
typedef enum
{
    LClientServer_NotificationFlags_OVERFLOW     = 0x80,    // An overflow of the notification occurred and
                                                            //  previous notifications were missed.
} LClientServer_NotificationFlags;


/************************************************************************************************************\

Structure:      LClientServer_CreateAttributes

Description:    Client server object creation attributes.

Comments:       LClientServer_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLClientServer_CreateAttributes
{
    LClientServer_AttributeType eType;              // Shall be LClientServer_AttributeType_CREATE
    MUINT8                      uiServerName;       // The numerical name of the server
    MUINT8                      auiServerVersion[8]; // Return the server version numbers
    MUINT32                     uiTimeout_ms;
} LClientServer_CreateAttributes;

/************************************************************************************************************\

Structure:      LClientServer_Notification

Description:    Client and server notification.

Comments:       - A number of notification data bytes may follow the notification structure.
                - The value oHeader.uSize shall be a multiple of 8 bytes.

\************************************************************************************************************/
typedef struct tagLClientServer_NotificationHeader
{
    MUINT8                     uiSize;   // Total size of the notification header with extra data following
    //  the header structure. This total size doesn't include the
    //  initial eType member.
    MUINT8                     uiRsvd0;  // uClient
    MUINT8                     uiFlags;  // A combination of NotificationFlags_* values.
    MUINT8                     uiRsvd1;  // uServer
    MINT32                     iError;   // Error code.
    MUINT64                    uiData;   // Single data associated with the notification.
    MUINT64                    uiTag;    // Tag associated with the notification.
} LClientServer_NotificationHeader;

typedef struct tagLClientServer_Notification
{
    LClientServer_NotificationType eType;     // One of the NotificationType_* enum.

    LClientServer_NotificationHeader oNotHeader;

} LClientServer_Notification;

/************************************************************************************************************\


\************************************************************************************************************/

typedef struct tagLClientServer_SubmissionHeader
{
    MUINT8                     uiSize;   // Total size of the notification header with extra data following
    //  the header structure. This total size doesn't include the
    //  initial eType member.
    MUINT8                     uiRsvd0;  // uClient.
    MUINT8                     uiFlags;  // A combination of NotificationFlags_* values.
    MUINT8                     uiRsvd1;  // uServer
    MUINT8                     uiMode;
    MUINT8                     uiParam;
    MUINT8                     uiValue;
    MUINT64                    uiData;   // Single data associated with the notification.
    MUINT64                    uiTag;    // Tag associated with the notification.
} LClientServer_SubmissionHeader;

/************************************************************************************************************\

Structure:      LClientServer_Submission

Description:    Client and server notification.

Comments:       - A number of notification data bytes may follow the notification structure.
                - The value oHeader.uSize shall be a multiple of 8 bytes.

\************************************************************************************************************/
typedef struct tagLClientServer_Submission
{
    LClientServer_SubmissionType eType;   // One of the NotificationType_* enum.

    LClientServer_SubmissionHeader oSubHeader;

} LClientServer_Submission;

#endif //LCLIENTSERVERSHAREDENUMS

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LClient_Create

Description:    Creates a client object.

Parameters:     IN  hDev                    Handle of the device for which to create a client object.
                IN  poAttributes            Pointer to the attribute type member of a CreateAttributes 
                                             structure describing the attributes of the client object to 
                                             create. NULL indicates the use of the default attributes.
                OUT phClient                Pointer to receive the handle of the client object created.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_RESOURCES    No more client-server channels available.

Comments:       - This method creates a client object for communicating with a "named" server running 
                  on a remote processor. A device may have up to LCLIENT_MAX_SERVERS servers running 
                  and up to LCLIENT_MAX_CLIENTS connected to those servers.
                - Multiple clients may connect to the same server.
                - The client receives the server version number in the auServerVersion member of the 
                  attributes structure. The client shall use this information to adapt to the capabilities 
                  of the server and establish a proper communication protocol.

\************************************************************************************************************/
LAPI LStatus LClient_Create(
                LDevice_Handle                   hDev,
                LClientServer_AttributeType*     poAttributes,
                LClient_Handle*                  phClient);

/************************************************************************************************************\

Function:       LClient_AddRef

Description:    Adds a reference on a client object.

Parameters:     IN  hClient                 Handle of a client object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - The AddRef / UnRef methods provide the necessary reference counting mechanism for sharing 
                  an object among a number of modules and making sure the object is destroyed when no 
                  reference remains.
                - This method adds a reference on a device thread object. The object exists until the number 
                  of references on it reaches zero (0). The methods UnRef and Destroy decrement the reference 
                  count.
                - Creating an object sets an initial reference on the object.

\************************************************************************************************************/
LAPI LStatus LClient_AddRef(
                LClient_Handle       hClient);

/************************************************************************************************************\

Function:       LClient_GetRefCount

Description:    Gets the number of references on a client object.

Parameters:     IN  hClient                 Handle of a client object.
                OUT MUINT32                 Pointer to an unsigned integer receiving the number
                                             of references on this object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.

Comments:       - This method returns the current number of references on a device thread object.
                - The value returned is always larger than 0. Otherwise, the object wouldn't exist.

\************************************************************************************************************/
LAPI LStatus LClient_GetRefCount(
                LClient_Handle    hClient);

/************************************************************************************************************\

Function:       LClient_UnRef

Description:    Removes a reference on a client object.

Parameters:     IN  hClient                 Handle of a client object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - The AddRef / UnRef methods provide the necessary reference counting mechanism for sharing 
                  an object among a number of modules and making sure the object is destroyed when no 
                  reference remains.
                - If multiple references remain on the client object, this method removes a reference. 
                  Otherwise, it performs a Destroy operation on the client object.
                - An application can use either of the UnRef or Destroy methods to destroy a client 
                  object. We recommend matching an UnRef operation with an AddRef operation, and a Destroy 
                  operation with a Create operation 

\************************************************************************************************************/
LAPI LStatus LClient_UnRef(
                LClient_Handle    hClient);

/************************************************************************************************************\

Function:       LClient_Destroy

Description:    Destroys a device thread object.

Parameters:     IN  hClient                 Handle of a client object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - If a single reference remains on the client object, this method destroys the object 
                  and the associated system resources. Otherwise, it performs an UnRef operation on the device 
                  thread object.
                - Remaining submissions and notifications in the client-server queues are flushed when the 
                  destroy operation takes place. An application should synchronize with the completion of the 
                  last submission to a server before calling this method.

\************************************************************************************************************/
LAPI LStatus LClient_Destroy(
                LClient_Handle    hClient);

/************************************************************************************************************\

Function:       LClient_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hClient                 Handle of a client object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes. 
                                            On input, the type member describes the type of the structure. 
                                            On output, the remaining members of the structure are filled with 
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default 
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is 
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LClient_GetDefaultAttributes(
                LClient_Handle                  hClient,
                LClientServer_AttributeType*    peAttributes );

/************************************************************************************************************\

Function:       LClient_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hClient                 Handle of a client object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes. 
                                            On input, the type member describes the type of the structure. 
                                            On output, the remaining members of the structure are filled with 
                                            the current value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with the 
                  current values.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is 
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LClient_GetAttributes(
                LClient_Handle                   hClient,
                LClientServer_AttributeType*     peAttributes);

/************************************************************************************************************\

Function:       LClient_SendSubmission

Description:    Sends a submission to the server.

Parameters:     IN  hClient                 Handle of a client object.
                IN  poSubmission            Pointer to the submission type member of a Submission structure 
                                            describing the submission to send to the server.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method sends a submission to a connected server. The tag associated with the submission 
                  can be retrieved using the GetSubmissionTag API.
                - The timeout applies to the operation of transmitting the request to the server. Once in 
                  the server submission queue, a request cannot timeout. The client should restart the server 
                  if it is not responding.
\************************************************************************************************************/
LAPI LStatus LClient_SendSubmission(
                LClient_Handle                  hClient,
                LClientServer_SubmissionType*   poSubmission,
                MUINT32                         uiTimeout_ms );

/************************************************************************************************************\

Function:       LClient_GetSubmissionTag

Description:    Gets the tag of the last submission to the server.

Parameters:     IN  hClient                 Handle of a client object.
                OUT puiLastSubmissionTag    Pointer to an unsigned integer to receive the value of the last
                                            submission tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission issued to the server.
                - A client object maintains a pseudo-timeline of the submissions flowing through its 
                  submission queue. A submission tag is associated with each submission received by the client  
                  object. Once a submission is completed, its submission tag becomes a completion 
                  tag.

\************************************************************************************************************/
LAPI LStatus LClient_GetSubmissionTag(
                LClient_Handle    hClient,
                MUINT64*          puiLastSubmissionTag);

/************************************************************************************************************\

Function:       LClient_GetCompletionTag

Description:    Gets the tag of the last completion received from the server.

Parameters:     IN  hClient                 Handle of a client object.
                OUT puiLastCompletionTag    Pointer to an unsigned integer to receive the value of the last
                                            completion tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission completed by the server.

\************************************************************************************************************/
LAPI LStatus LClient_GetCompletionTag(
                LClient_Handle    hClient,
                MUINT64*          puiLastCompletionTag);

/************************************************************************************************************\

Function:       LClient_WaitCompletionTag

Description:    Waits for the tag of a completion from the server.

Parameters:     IN  hClient                 Handle of a client object.
                IN  uiCompletionTag         Completion tag.
                IN  uiTimeout_ms            Timeout in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method is used to synchronize a client CPU thread with a given submission sent to the 
                  server. The CPU thread sleeps until the submission associated with the given tag is 
                  complete or a timeout occurs. The waiting terminates when the given tag becomes smaller 
                  than or equal to the last completion tag received form the server or a timeout occurs.
                - The value ~0 (0xffffffffffffffff) for the uCompletionTag parameter indicates to wait for 
                  the last submission tag. This operation results in waiting for all the previous submissions 
                  to complete.
                - A timeout of 0 verifies the completion status immediately without sleeping. The error 
                  LStatus_TIMEOUT is returned if the submission hasn't completed.

\************************************************************************************************************/
LAPI LStatus LClient_WaitCompletionTag(
                LClient_Handle    hClient,
                MUINT64           uiCompletionTag,
                MUINT32           uiTimeout_ms);

/************************************************************************************************************\

Function:       LClient_WaitNotification

Description:    Waits for a notification from the server.

Parameters:     IN  hClient                 Handle of a client object.
                OUT poNotification          Pointer to the notification type member of a notification structure. 
                                            On input, the type member describes the type of the structure and 
                                            the available size. 
                                            On output, all the members of the structure are filled with the 
                                            information of the notification and a new type is returned.
                IN  uiTimeout_ms             Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method is used to synchronize a client CPU thread with the arrival of a notification 
                  from the server. The CPU thread sleeps until a notification is received or a timeout occurs.
                - The application provides a notification structure on input where the type member shall be 
                  NotificationType_HEADER. When a notification becomes available, the corresponding 
                  notification header structure is filled and the type and size members are updated to
                  match the current notification. Only the header of the current notification is retrieved 
                  from the notification queue when this method is called. It gives the application the opportunity 
                  to inspect the notification type and the size of the notification before it retrieves the 
                  associated data using the GetNotificationData method.  
                - An application shall monitor its notification queue regularly to prevent it from becoming 
                  full in which case it may miss notifications from the server. In the event of a full 
                  notification queue, the server would tag the next notification with the flag 
                  NotificationFlags_OVERFLOW to indicate that some previous notifications were missed.
                - A timeout of 0 verifies the notification status immediately without sleeping. The error 
                  LStatus_TIMEOUT is returned if a notification hasn't been received.
                - Calling this method will flush pending data associated with a previous notification that 
                  wasn't retrieved by calling GetNotificationData.
                - The application shall synchronize the access to a notification queue such that a CPU thread 
                  retrieving a notification will get both the header and the data before another CPU thread is 
                  allowed to access the notification queue.

\************************************************************************************************************/
LAPI LStatus LClient_WaitNotification(
                LClient_Handle                      hClient,
                LClientServer_NotificationType*     poNotification,
                MUINT32                             uiTimeout_ms);

/************************************************************************************************************\

Function:       LClient_GetNotificationData

Description:    Gets the data associated with the current notification.

Parameters:     IN  hClient                 Handle of a client object.
                IN  uiSize                   Size in bytes of the output data the buffer.
                IN  puiData                  Pointer to a buffer of bytes to receive the data.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method retrieves the data associated with the current notification.

                - An application isn't required to retrieve all or any of the data associated with a 
                  notification. A call to WaitNotification flushes any pending data from the current 
                  notification.

\************************************************************************************************************/
LAPI LStatus LClient_GetNotificationData(
                LClient_Handle     hClient,
                MUINT32            uiSize,
                MUINT8*            puiData);

/************************************************************************************************************\

Function:       LClient_Restart

Description:    Restarts the server.

Parameters:     IN  hClient                 Handle of a client object.       

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method restarts the connected server.
                - The server is stopped, submission queue is flushed and a notification is sent to the client
                  through the notification queue indicating the tag of the last submission retrieved by the 
                  server. The flushed submissions will never complete and proper actions by the client are 
                  required to fully recover.
                - This API is blocking and won't return until the system processes the request. A 
                  NotificationType_RESTART notification will be available from the notification queue to 
                  indicate the tag of the last submission retrieved by the server.

\************************************************************************************************************/
LAPI LStatus LClient_Restart(
                LClient_Handle     hClient);


#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LCIENT_H
